function P = scheme3(h, g, Pc, Pn, QoS_ir, QoS_ri)
%% optimal solution with power splitting ratio 0.5
% Read this program together with the conference paper in ICASSP'16
% S. Wang, Y.-C. Wu, and M. Xia, Achieving global optimality in wirelessly-powered multi-antenna TWRC with lattice codes, in Proc. IEEE ICASSP16, Shang Hai, China, Mar. 2016, pp. 3556-3560.
% This function solves the semidefinite programing (SDP) problem P3 using CVX.

betai=0.5*ones(2,1); % set the power splitting ratio to 0.5

eta=0.7; % power conversion efficienty for wireless power transfer
sigma=sqrt(Pn); % square root of noise power

%% Calculate SINR QoS
alpha=QoS_ir-QoS_ir./(QoS_ir(1)+QoS_ir(2)); % \alpha_i defined in eqn. 12: uplink SINR QoS [Property 1]
theta=QoS_ri-1; % \theta_i defined in problem P2: downlink SINR QoS

%% Reduce the problem dimension using Property 2
G=[g(:,:,1) g(:,:,2)]; % G defined under problem P2: subspace for transmit beamformer v
Phi=zeros(2,2,2); % \Phi defined under problem P2
for i=1:2
    Phi(:,:,i)=G'*g(:,:,i)*g(:,:,i)'*G;
end
H=[h(:,:,1) h(:,:,2)]; % H defined under problem P2: subspace for receive beamformer w
Xi=zeros(2,2,2); % \Xi defined under problem P2
for i=1:2
    Xi(:,:,i)=H'*h(:,:,i)*h(:,:,i)'*H;
end

%% Solve the SDP problem P3
cvx_begin sdp quiet
cvx_precision best

variable A(2,2) hermitian % A defined under problem P2: coefficient matrix for v
variable B(2,2) hermitian % B defined under problem P2: coefficient matrix for w
variable xii(2) % slack variables
variable mui(2) % slack variables
variable phii(2) % slack variables

minimize real(trace(G'*G*A)+trace_inv(xii(1))+trace_inv(xii(2))); % totol transmit power
subject to

for i=1:2
    real(trace(Xi(:,:,i)*B))==alpha(i)*Pn*xii(i); % (17a) uplink data-rate QoS constraint
end

for i=1:2
    [real(trace(Phi(:,:,i)*A))./theta(i)-Pn sigma;sigma betai(i)]>=0; % (17b) downlink data-rate QoS constraint
end

for i=1:2
    [eta*real(trace(Phi(:,:,i)*A)+Pn) mui(i);mui(i) 1-betai(i)]>=0; % (17c) energy harvesting constraint
end

for i=1:2
    mui(i)>=norm([phii(i) sqrt(2*Pc)]); % (17d) second-order cone constraint
    xii(i)>=pow_p(phii(i),-2); % (17d) power function of a scalar
end

trace(H'*H*B)==1; % (17e) norm constraint
A>=0; % (17e) positive semidefinite constraint
B>=0; % (17e) positive semidefinite constraint

cvx_end
% Compute the objective value of P3
P = real(trace(G'*G*A))+1./xii(1)+1./xii(2);

end

